import React, { useState, useEffect } from "react";
import api from "../api";
import { Layout } from "../components/Layout";
import { StatCard } from "../components/StatCard";
import { DataTable } from "../components/DataTable";
import { LeaveModal } from "../components/LeaveModal";
import {
  FaCalendarCheck,
  FaClock,
  FaTimesCircle,
  FaPlus,
} from "react-icons/fa";

export const Employee = () => {
  const [leaves, setLeaves] = useState([]);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [loading, setLoading] = useState(false);
  const [form, setForm] = useState({ from: "", to: "", reason: "" });
  const token = localStorage.getItem("token");
  let role = null;
  if (token) {
    try {
      role = JSON.parse(atob(token.split(".")[1])).role;
    } catch (e) {
      console.error("Invalid token", e);
    }
  }

  const fetchLeaves = async () => {
    try {
      const res = await api.get("/leave/me");
      setLeaves(res.data);
    } catch (err) {
      console.log(err);
    }
  };

  useEffect(() => {
    fetchLeaves();
  }, []);

  const applyLeave = async (e) => {
    e.preventDefault();
    setLoading(true);
    if (role === "employee") {
      try {
        await api.post("/leave/apply", form);
        fetchLeaves();
        setIsModalOpen(false);
        setForm({ from: "", to: "", reason: "" });
      } catch (err) {
        console.log(err);
      } finally {
        setLoading(false);
      }
    } else {
      window.location.href = "/login";
    }
  };

  const columns = [
    {
      header: "From",
      accessor: "from",
      render: (row) => new Date(row.from).toLocaleDateString(),
    },
    {
      header: "To",
      accessor: "to",
      render: (row) => new Date(row.to).toLocaleDateString(),
    },
    { header: "Reason", accessor: "reason" },
    {
      header: "Status",
      accessor: "status",
      render: (row) => (
        <span
          className={`px-3 py-1 rounded-full text-xs font-semibold ${
            row.status === "approved"
              ? "bg-green-100 text-green-700"
              : row.status === "rejected"
                ? "bg-red-100 text-red-700"
                : "bg-yellow-100 text-yellow-700"
          }`}
        >
          {row.status.charAt(0).toUpperCase() + row.status.slice(1)}
        </span>
      ),
    },
  ];

  // Calculate stats
  const approvedCount = leaves.filter((l) => l.status === "approved").length;
  const pendingCount = leaves.filter((l) => l.status === "pending").length;
  const rejectedCount = leaves.filter((l) => l.status === "rejected").length;

  return (
    <Layout>
      <div className="space-y-6">
        <div className="flex justify-between items-center">
          <h2 className="text-2xl font-bold text-gray-800">My Leaves</h2>
          <button
            onClick={() => setIsModalOpen(true)}
            className="flex items-center space-x-2 bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded-lg shadow-md transition-colors duration-200"
          >
            <FaPlus />
            <span>Apply Leave</span>
          </button>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
          <StatCard
            title="Approved"
            value={approvedCount}
            icon={<FaCalendarCheck />}
            color="bg-green-500"
          />
          <StatCard
            title="Pending"
            value={pendingCount}
            icon={<FaClock />}
            color="bg-yellow-500"
          />
          <StatCard
            title="Rejected"
            value={rejectedCount}
            icon={<FaTimesCircle />}
            color="bg-red-500"
          />
        </div>

        <DataTable columns={columns} data={leaves} />

        <LeaveModal
          isOpen={isModalOpen}
          onClose={() => setIsModalOpen(false)}
          onSubmit={applyLeave}
          form={form}
          setForm={setForm}
          loading={loading}
        />
      </div>
    </Layout>
  );
};
